<?php
declare(strict_types=1);

/**
 * modules/admin/settings.php
 *
 * Stores company-wide settings:
 *  - Quote T&Cs (used by quotes/pdf.php)
 *  - Quote watermark text (used in preview; your PDF already uses company name)
 *
 * Parts:
 *  1) Bootstrap & guards
 *  2) Read current settings
 *  3) Handle POST save
 *  4) Render form + live preview (with watermark)
 */

////////////////////////////////////////////////////////////////
// 1) Bootstrap & guards
////////////////////////////////////////////////////////////////
require_once dirname(__DIR__, 2) . '/config/functions.php';
require_role(['Admin']);

$user = current_user();
$cid  = (int)($user['company_id'] ?? 0);
if ($cid <= 0) redirect(url_public('login.php'));

$errors = [];
$notice = null;

////////////////////////////////////////////////////////////////
// 2) Read current settings
////////////////////////////////////////////////////////////////
function load_setting(int $cid, string $key): string {
  $stmt = db()->prepare("SELECT `value` FROM settings WHERE company_id=:cid AND `key`=:k LIMIT 1");
  $stmt->execute([':cid'=>$cid, ':k'=>$key]);
  $row = $stmt->fetch();
  return (string)($row['value'] ?? '');
}
$curTerms     = load_setting($cid, 'quote_terms');
$curWatermark = load_setting($cid, 'quote_watermark') ?: 'Brownhill Group Limited';

////////////////////////////////////////////////////////////////
// 3) Handle POST save
////////////////////////////////////////////////////////////////
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  try {
    csrf_verify((string)($_POST['csrf'] ?? ''));

    $terms     = trim((string)($_POST['quote_terms'] ?? ''));
    $watermark = trim((string)($_POST['quote_watermark'] ?? 'Brownhill Group Limited'));
    if ($watermark === '') $watermark = 'Brownhill Group Limited';

    // upsert settings
    $ins = db()->prepare("
      INSERT INTO settings (company_id, `key`, `value`, updated_at)
      VALUES (:cid, :k, :v, NOW())
      ON DUPLICATE KEY UPDATE `value` = VALUES(`value`), updated_at = NOW()
    ");
    $ins->execute([':cid'=>$cid, ':k'=>'quote_terms', ':v'=>$terms]);
    $ins->execute([':cid'=>$cid, ':k'=>'quote_watermark', ':v'=>$watermark]);

    audit_log('settings.update','settings',0,['keys'=>['quote_terms','quote_watermark']]);
    $notice     = 'Settings saved.';
    $curTerms   = $terms;
    $curWatermark = $watermark;
  } catch (Throwable $e) {
    $errors[] = (defined('APP_ENV') && APP_ENV==='dev') ? $e->getMessage() : 'Unable to save settings.';
  }
}

////////////////////////////////////////////////////////////////
// 4) Render
////////////////////////////////////////////////////////////////
include dirname(__DIR__, 2) . '/includes/header.php';
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <h1 class="h4 mb-0">System Settings</h1>
</div>

<?php if ($notice): ?><div class="alert alert-success"><?= e($notice) ?></div><?php endif; ?>
<?php if ($errors): ?><div class="alert alert-danger"><ul class="mb-0"><?php foreach($errors as $er) echo '<li>'.e($er).'</li>'; ?></ul></div><?php endif; ?>

<form method="post" id="settingsForm" autocomplete="off">
  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">

  <div class="row g-3">
    <div class="col-12 col-lg-7">
      <div class="card shadow-sm h-100">
        <div class="card-body">
          <h2 class="h6 text-uppercase text-muted mb-3">Quote Terms &amp; Conditions</h2>

          <div class="d-flex gap-2 mb-2">
            <button type="button" class="btn btn-outline-secondary btn-sm" id="btnLoadGeneric">Load Generic Sample</button>
            <button type="button" class="btn btn-outline-primary btn-sm" id="btnLoadBrownhill">Load Brownhill T&amp;Cs</button>
          </div>

          <textarea class="form-control" name="quote_terms" rows="14" placeholder="Paste or write your T&Cs here..."><?= e($curTerms) ?></textarea>

          <hr class="my-3">

          <div class="row g-2">
            <div class="col-md-8">
              <label class="form-label">Watermark Text</label>
              <input class="form-control" name="quote_watermark" id="wmText" value="<?= e($curWatermark) ?>">
              <div class="form-text">Used in the on-screen preview below. Your PDFs will continue to use your company name as watermark.</div>
            </div>
            <div class="col-md-4 d-grid">
              <button class="btn btn-dark mt-4">Save Settings</button>
            </div>
          </div>
        </div>
      </div>
    </div>

    <!-- Live Preview with watermark -->
    <div class="col-12 col-lg-5">
      <div class="card shadow-sm h-100">
        <div class="card-body">
          <h2 class="h6 text-uppercase text-muted mb-3">Live Preview</h2>
          <div class="border rounded position-relative overflow-auto" style="height: 480px; background:#fff;">
            <div id="watermark" class="wm"></div>
            <pre id="termsPreview" class="p-3 mb-0" style="white-space: pre-wrap; font-family: system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial, sans-serif; font-size: .92rem; line-height: 1.35;">
<?= e($curTerms ?: '— No T&Cs set —') ?>
            </pre>
          </div>
          <div class="form-text mt-2">This is only a preview. The PDF layout is handled by <code>modules/quotes/pdf.php</code>.</div>
        </div>
      </div>
    </div>
  </div>
</form>

<style>
/* Watermark overlay */
.wm::before{
  content: attr(data-text);
  position: absolute;
  inset: 0;
  display: block;
  font-weight: 700;
  font-size: 48px;
  color: #0f172a;            /* slate-900 base */
  opacity: .06;              /* very subtle */
  text-align: center;
  transform: rotate(-28deg);
  transform-origin: center;
  letter-spacing: 2px;
  user-select: none;
  pointer-events: none;
  white-space: pre-wrap;
}
/* Make watermark cover the preview nicely */
.wm{
  position:absolute;
  inset:0;
}
</style>

<script>
(function(){
  const $ta      = document.querySelector('textarea[name="quote_terms"]');
  const $preview = document.getElementById('termsPreview');
  const $btnGen  = document.getElementById('btnLoadGeneric');
  const $btnBH   = document.getElementById('btnLoadBrownhill');
  const $wm      = document.getElementById('watermark');
  const $wmText  = document.getElementById('wmText');

  function syncPreview(){
    const txt = ($ta.value || '').trim();
    $preview.textContent = txt ? txt : '— No T&Cs set —';
  }
  function syncWatermark(){
    const wm = ($wmText.value || 'Brownhill Group Limited').trim();
    $wm.setAttribute('data-text', wm);
  }

  $ta.addEventListener('input', syncPreview);
  $wmText.addEventListener('input', syncWatermark);

  // Generic sample
  $btnGen.addEventListener('click', function(){
    const sample = [
      '1. All prices are GBP and exclude VAT unless stated.',
      '2. Waiting time, parking and tolls may apply where relevant.',
      '3. Cancellations within 24 hours may be charged in full.',
      '4. Liability is limited to the value of the booking.',
      '5. Payment terms: 14 days unless otherwise agreed.',
      '6. Full terms available on request.'
    ].join('\\n');
    $ta.value = sample;
    syncPreview();
  });

  // Brownhill Group Limited — T&Cs (from your document)
  $btnBH.addEventListener('click', function(){
    const bh = [
      'COMPLIMENTARY SERVICES',
      '• 60 mins waiting time for airport pickups (inbound)',
      '• 15 mins for all other pickups (outbound)',
      '• Meet & Greet, Flight Monitoring, Onboard Wi-Fi (Business/First Class vehicles)',
      '• Bottled Water (Business/First Class vehicles), Luggage Assistance, Chauffeur monitored by us.',
      '',
      'WAITING TIME (per hour after complimentary period)',
      '• Business Sedan: £50 | First Class Sedan: £55 | Business Van: £55',
      '• Economy Sedan: £40 | Economy Van MPV5: £45 | Economy Mini Bus: £50',
      '',
      'AIRPORT PARKING CHARGES',
      '• Paid in addition to booking fee.',
      '• Options: Pay driver in cash OR request a secure payment link.',
      '• Original parking receipt provided by driver.',
      '',
      'CONGESTION CHARGES',
      '• If journey goes through a congestion zone, £7.50 is chargeable to the client.',
      '',
      'ADDITIONAL DROPOFF CHARGES',
      '• Business Sedan £10 | First Class Sedan £15 | Business Van £15',
      '• Economy Sedan £7 | Economy Van MPV5 £9 | Economy Mini Bus £12',
      '',
      'CANCELLATION POLICY',
      '• Free cancellation if notified at least 24 hours before pick-up.',
      '• Cancellations within 24 hours may be charged in full.',
      '',
      'DRIVER INFORMATION',
      '• Chauffeur details provided via email 12 hours before service time.',
      '',
      'MEETING POINTS',
      '• Arrivals: Chauffeur greets at the arrival hall with a name board.',
      '• Departures: Pick-up from hotel main lobby.',
      '',
      'PAYMENT',
      '• Pay by payment link, BACS or bank transfer (details on invoice).',
      '• Payment link incurs a 3% processing fee; bank transfer has no extra fee.',
      '• Late payments: 1% per month interest (calculated daily) may apply. Service may be suspended until overdue amounts are settled.',
      '',
      'DISCLAIMER',
      '• Final service fee may vary slightly due to unforeseen circumstances.',
      '• Brown Hill Chauffeurs is not liable for delays caused by factors beyond our control (e.g., traffic).',
      '',
      'MODIFICATIONS',
      '• Terms & Conditions may be updated from time to time; changes will be communicated promptly.',
      '',
      'CONTACT',
      '• Phone: +44 (0)208 064 2662',
      '• Email: info@brown-hillchauffeurs.co.uk',
      '• Web: www.brown-hillchauffeurs.co.uk'
    ].join('\\n');
    $ta.value = bh;
    syncPreview();
    if (!$wmText.value.trim()) {
      $wmText.value = 'Brownhill Group Limited';
    }
    syncWatermark();
  });

  // initial state
  syncPreview();
  syncWatermark();
})();
</script>

<?php include dirname(__DIR__, 2) . '/includes/footer.php'; ?>
