<?php
declare(strict_types=1);

/**
 * modules/payments/drivers.php
 *
 * Driver Payouts — review & mark bookings as paid to drivers.
 * - Cleared jobs only (status = Completed).
 * - Unpaid/Paid filters rely on outstanding (= driver_due - paid_sum).
 * - Modal to mark driver payout.
 *
 * Added earlier:
 *  - Accounting row into `transactions` (DRIVER_PAYOUT, Chauffeur, expense)
 *  - Upsert/remove `pending_payments` (type='driver') for partial/clear
 *  - Redirect to printable receipt after saving payout
 *
 * NEW:
 *  - "View Job" button + modal with client pricing & payments summary
 *  - AJAX endpoints handled BEFORE any output to avoid header warnings
 */

require_once dirname(__DIR__, 2) . '/config/functions.php';
require_role(['MD','Accounts','Admin','Management']);

$user = current_user();
$cid  = (int)($user['company_id'] ?? 0);
if ($cid <= 0) redirect(url_public('login.php'));

/* ---------------- Small helpers (guarded) ---------------- */
if (!function_exists('tbl_exists')) {
  function tbl_exists(string $t): bool {
    try { db()->query("SELECT 1 FROM `{$t}` LIMIT 1"); return true; } catch (Throwable $e) { return false; }
  }
}
if (!function_exists('decode_bank')) {
  function decode_bank(?string $json): array {
    $defaults = ['bank_name'=>'','account_name'=>'','account_number'=>'','sort_code'=>'','iban'=>'','notes'=>''];
    if (!$json) return $defaults;
    $arr = json_decode($json, true);
    return is_array($arr) ? array_merge($defaults, array_intersect_key($arr, $defaults)) : $defaults;
  }
}
if (!function_exists('n2')) {
  function n2($v): string { return number_format((float)$v, 2); }
}
/* DO NOT redefine e() if your project already has it in functions.php */
if (!function_exists('e')) {
  function e($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

/** Sum of driver payouts already made for a booking */
function driver_paid_sum(int $companyId, int $bookingId): float {
  try {
    $q = db()->prepare("SELECT COALESCE(SUM(amount),0) FROM driver_payouts WHERE company_id=:cid AND booking_id=:bid");
    $q->execute([':cid'=>$companyId, ':bid'=>$bookingId]);
    return (float)$q->fetchColumn();
  } catch (Throwable $e) { return 0.0; }
}

/** Server-side compute of driver_due for a booking */
function booking_driver_due(int $companyId, int $bookingId): array {
  $q = db()->prepare("
    SELECT booking_ref,
           (COALESCE(total_driver_price,0)+COALESCE(driver_parking_fee,0)+COALESCE(driver_waiting_fee,0)) AS driver_due
      FROM bookings
     WHERE company_id=:cid AND id=:bid
     LIMIT 1
  ");
  $q->execute([':cid'=>$companyId, ':bid'=>$bookingId]);
  $row = $q->fetch();
  return [
    'booking_ref' => (string)($row['booking_ref'] ?? ('#'.$bookingId)),
    'driver_due'  => (float)($row['driver_due']  ?? 0.0),
  ];
}

/** Sum of client payments received for a booking */
function client_paid_sum(int $companyId, int $bookingId): float {
  if (tbl_exists('transactions')) {
    try {
      $q = db()->prepare("
        SELECT COALESCE(SUM(amount),0)
          FROM transactions
         WHERE company_id=:cid
           AND booking_id=:bid
           AND account_code='CHAUF_INCOME'
           AND type='income'
      ");
      $q->execute([':cid'=>$companyId, ':bid'=>$bookingId]);
      $v = (float)$q->fetchColumn();
      if ($v > 0) return $v;
    } catch (Throwable $e) {}
  }
  if (tbl_exists('invoice_payments') && tbl_exists('invoices')) {
    try {
      $q = db()->prepare("
        SELECT COALESCE(SUM(ip.amount),0)
          FROM invoice_payments ip
          JOIN invoices i ON i.id = ip.invoice_id
         WHERE ip.company_id=:cid
           AND i.company_id = :cid
           AND i.booking_id = :bid
      ");
      $q->execute([':cid'=>$companyId, ':bid'=>$bookingId]);
      return (float)$q->fetchColumn();
    } catch (Throwable $e) {}
  }
  return 0.0;
}

/* ------------------------------------------------------------------
   AJAX micro-endpoints — HANDLE BEFORE ANY HTML OUTPUT
   ------------------------------------------------------------------ */

/* A) Bank details */
if (($_GET['fetch_bank'] ?? '') === '1' && isset($_GET['driver_id'])) {
  try {
    if (!headers_sent()) header('Content-Type: text/html; charset=utf-8');
    $did = (int)$_GET['driver_id'];

    $q = db()->prepare("SELECT name, bank_details FROM drivers WHERE id=:id AND company_id=:cid LIMIT 1");
    $q->execute([':id' => $did, ':cid' => $cid]);

    if ($d = $q->fetch()) {
      $b = decode_bank((string)($d['bank_details'] ?? ''));
      ?>
      <table class="table table-sm align-middle mb-0">
        <tbody>
          <tr><th style="width:35%;">Driver</th><td><?= e($d['name'] ?? '') ?></td></tr>
          <tr><th>Bank</th><td><?= e($b['bank_name'] ?: '—') ?></td></tr>
          <tr><th>Account Holder</th><td><?= e($b['account_name'] ?: '—') ?></td></tr>
          <tr><th>Account Number</th><td><?= e($b['account_number'] ?: '—') ?></td></tr>
          <tr><th>Sort Code</th><td><?= e($b['sort_code'] ?: '—') ?></td></tr>
          <tr><th>IBAN</th><td><?= e($b['iban'] ?: '—') ?></td></tr>
          <tr><th>Notes</th><td><?= nl2br(e($b['notes'] ?: '—')) ?></td></tr>
        </tbody>
      </table>
      <?php
    } else {
      echo '<div class="text-muted">Not found.</div>';
    }
  } catch (Throwable $e) {
    http_response_code(500);
    echo (defined('APP_ENV') && APP_ENV === 'dev') ? e($e->getMessage()) : 'Error';
  }
  exit;
}

/* B) Job details (client pricing + payments) */
if (($_GET['fetch_job'] ?? '') === '1' && isset($_GET['booking_id'])) {
  try {
    if (!headers_sent()) header('Content-Type: text/html; charset=utf-8');
    $bid = (int)$_GET['booking_id'];

    $q = db()->prepare("
      SELECT
        booking_ref, client_name, client_phone, client_email,
        pickup_date, pickup_time, pickup_address, dropoff_address, via,
        total_client_price, client_parking_fee, client_waiting_fee,
        notes
      FROM bookings
      WHERE id=:bid AND company_id=:cid
      LIMIT 1
    ");
    $q->execute([':bid'=>$bid, ':cid'=>$cid]);
    $b = $q->fetch();

    if (!$b) { echo '<div class="text-danger">Booking not found.</div>'; exit; }

    $ref     = (string)($b['booking_ref'] ?? ('#'.$bid));
    $pdt     = trim((string)($b['pickup_date'] ?? '') . ' ' . (string)($b['pickup_time'] ?? ''));
    $client  = (string)($b['client_name'] ?? '—');

    $cp      = (float)($b['total_client_price'] ?? 0);
    $cpark   = (float)($b['client_parking_fee'] ?? 0);
    $cwait   = (float)($b['client_waiting_fee'] ?? 0);
    $ctotal  = round($cp + $cpark + $cwait, 2);

    $cpaid   = round(client_paid_sum($cid, $bid), 2);
    $cbal    = max(0.0, round($ctotal - $cpaid, 2));

    $via     = (string)($b['via'] ?? '');
    $from    = (string)($b['pickup_address'] ?? '');
    $to      = (string)($b['dropoff_address'] ?? '');
    $note    = (string)($b['notes'] ?? '');

    ?>
    <div class="mb-2">
      <div class="d-flex justify-content-between">
        <div>
          <div class="fw-semibold">Booking: <?= e($ref) ?></div>
          <div class="text-muted small"><?= e($client) ?> — <?= e($pdt) ?></div>
        </div>
        <div class="text-end small">
          <div>Client Paid: <strong>£<?= e(n2($cpaid)) ?></strong></div>
          <div>Client Balance: <span class="badge text-bg-<?= $cbal>0?'danger':'success' ?>">£<?= e(n2($cbal)) ?></span></div>
        </div>
      </div>
    </div>

    <div class="border rounded p-2 bg-light mb-3">
      <div class="row g-2">
        <div class="col-md-3"><div class="small text-muted">Client Price</div><div class="fw-semibold">£<?= e(n2($cp)) ?></div></div>
        <div class="col-md-3"><div class="small text-muted">Parking (Client)</div><div class="fw-semibold">£<?= e(n2($cpark)) ?></div></div>
        <div class="col-md-3"><div class="small text-muted">Waiting (Client)</div><div class="fw-semibold">£<?= e(n2($cwait)) ?></div></div>
        <div class="col-md-3"><div class="small text-muted">Client Total</div><div class="fw-bold">£<?= e(n2($ctotal)) ?></div></div>
      </div>
    </div>

    <div class="row g-2 mb-2 small">
      <div class="col-md-6"><span class="text-muted">Pickup</span><div><?= nl2br(e($from ?: '—')) ?></div></div>
      <div class="col-md-6"><span class="text-muted">Dropoff</span><div><?= nl2br(e($to   ?: '—')) ?></div></div>
      <?php if ($via !== ''): ?>
      <div class="col-12"><span class="text-muted">Via</span><div><?= nl2br(e($via)) ?></div></div>
      <?php endif; ?>
    </div>

    <?php if ($note !== ''): ?>
      <div class="mb-2">
        <div class="small text-muted">Note</div>
        <div class="border rounded p-2"><?= nl2br(e($note)) ?></div>
      </div>
    <?php endif; ?>
    <?php
  } catch (Throwable $e) {
    http_response_code(500);
    echo (defined('APP_ENV') && APP_ENV === 'dev') ? e($e->getMessage()) : 'Error';
  }
  exit;
}

/* ---------------- Page state ---------------- */
$errors = [];
$notice = null;

/* ---------------- Filters (ENHANCED) ---------------- */
$today      = new DateTimeImmutable('today');
$monthStart = $today->modify('first day of this month');
$monthEnd   = $today->modify('last day of this month');

$from        = (string)($_GET['from'] ?? $monthStart->format('Y-m-01'));
$to          = (string)($_GET['to']   ?? $monthEnd->format('Y-m-d'));
$status      = (string)($_GET['status'] ?? 'unpaid'); // unpaid|paid|all
$q           = trim((string)($_GET['q'] ?? ''));      // generic search: ref/client (kept)
$driverName  = trim((string)($_GET['driver_name'] ?? '')); // NEW: partial driver name
$job6        = preg_replace('/\D+/', '', (string)($_GET['job6'] ?? '')); // NEW: last 6 digits only
$outMinStr   = (string)($_GET['out_min'] ?? '');
$outMaxStr   = (string)($_GET['out_max'] ?? '');

if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $from)) $from = $monthStart->format('Y-m-d');
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $to))   $to   = $monthEnd->format('Y-m-d');
if (!in_array($status, ['unpaid','paid','all'], true)) $status = 'unpaid';

$outMin = ($outMinStr !== '' ? max(0.0, (float)$outMinStr) : null);
$outMax = ($outMaxStr !== '' ? max(0.0, (float)$outMaxStr) : null);

/* ---------------- Query bookings ---------------- */
$rows = [];
try {
  $hasPayouts = false;
  try { db()->query("SELECT 1 FROM driver_payouts LIMIT 1"); $hasPayouts = true; } catch (Throwable $ignore) {}

  $where = [
    "b.company_id = :cid",
    "b.pickup_date BETWEEN :from AND :to",
    "b.status = 'Completed'",
  ];
  $args = [':cid'=>$cid, ':from'=>$from, ':to'=>$to];

  // Keep original generic search (ref/client/driver)
  if ($q !== '') {
    $where[] = "(b.booking_ref LIKE :q OR b.client_name LIKE :q OR d.name LIKE :q)";
    $args[':q'] = '%'.$q.'%';
  }

  // NEW: dedicated partial driver name (case-insensitive LIKE)
  if ($driverName !== '') {
    $where[] = "d.name LIKE :driverName";
    $args[':driverName'] = '%'.$driverName.'%';
  }

  // NEW: last 6 digits of booking ref (if 6 digits supplied)
  if ($job6 !== '' && strlen($job6) === 6) {
    $where[] = "RIGHT(b.booking_ref, 6) = :job6";
    $args[':job6'] = $job6;
  }

  // Build HAVING for outstanding filters and status
  $havingParts = [];
  if ($status === 'unpaid')       $havingParts[] = 'outstanding > 0.0001';
  elseif ($status === 'paid')     $havingParts[] = 'outstanding < 0.0001';

  if ($outMin !== null) $havingParts[] = 'outstanding >= '.number_format($outMin, 2, '.', '');
  if ($outMax !== null) $havingParts[] = 'outstanding <= '.number_format($outMax, 2, '.', '');

  $sql = "
    SELECT
      b.id,
      b.booking_ref,
      b.client_name,
      b.pickup_date,
      b.pickup_time,
      b.status,

      COALESCE(b.total_driver_price,0) AS driver_price,
      COALESCE(b.driver_parking_fee,0) AS driver_parking,
      COALESCE(b.driver_waiting_fee,0) AS driver_waiting,

      (SELECT bv.driver_id
         FROM booking_vehicles bv
        WHERE bv.booking_id = b.id
        ORDER BY bv.sequence_order ASC, bv.id ASC
        LIMIT 1) AS driver_id,

      (SELECT d2.name
         FROM drivers d2
        WHERE d2.id = (SELECT bv2.driver_id
                         FROM booking_vehicles bv2
                        WHERE bv2.booking_id = b.id
                        ORDER BY bv2.sequence_order ASC, bv2.id ASC
                        LIMIT 1)
        LIMIT 1) AS driver_name,

      (SELECT d3.phone
         FROM drivers d3
        WHERE d3.id = (SELECT bv3.driver_id
                         FROM booking_vehicles bv3
                        WHERE bv3.booking_id = b.id
                        ORDER BY bv3.sequence_order ASC, bv3.id ASC
                        LIMIT 1)
        LIMIT 1) AS driver_phone,

      (COALESCE(b.total_driver_price,0) + COALESCE(b.driver_parking_fee,0) + COALESCE(b.driver_waiting_fee,0)) AS driver_due,

      ".($hasPayouts ? "
      (SELECT COALESCE(SUM(dp.amount),0)
         FROM driver_payouts dp
        WHERE dp.booking_id = b.id
          AND dp.company_id = b.company_id)
      " : "0")." AS paid_sum,

      ".($hasPayouts ? "
      (SELECT dpl.id
         FROM driver_payouts dpl
        WHERE dpl.booking_id = b.id
          AND dpl.company_id = b.company_id
        ORDER BY dpl.paid_at DESC, dpl.id DESC
        LIMIT 1)
      " : "NULL")." AS latest_payout_id,

      ".($hasPayouts ? "
      (SELECT dpl2.paid_at
         FROM driver_payouts dpl2
        WHERE dpl2.booking_id = b.id
          AND dpl2.company_id = b.company_id
        ORDER BY dpl2.paid_at DESC, dpl2.id DESC
        LIMIT 1)
      " : "NULL")." AS latest_paid_at

    FROM bookings b
    LEFT JOIN drivers d
      ON d.id = (SELECT bv4.driver_id
                   FROM booking_vehicles bv4
                  WHERE bv4.booking_id = b.id
                  ORDER BY bv4.sequence_order ASC, bv4.id ASC
                  LIMIT 1)
    WHERE ".implode(' AND ', $where)."
  ";

  $outer = "
    SELECT t.*,
           GREATEST(t.driver_due - t.paid_sum, 0) AS outstanding
      FROM ( $sql ) t
    ".(!empty($havingParts) ? 'HAVING '.implode(' AND ', $havingParts) : '')."
    ORDER BY t.pickup_date DESC, t.pickup_time DESC, t.id DESC
    LIMIT 1000
  ";

  $stmt = db()->prepare($outer);
  $stmt->execute($args);
  $rows = $stmt->fetchAll();

} catch (Throwable $e) {
  $errors[] = (defined('APP_ENV') && APP_ENV === 'dev') ? $e->getMessage() : 'Unable to load payouts.';
}

/* ---------------- POST: Save payout ---------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'mark_paid') {
  try {
    csrf_verify((string)($_POST['csrf'] ?? ''));
    $bid   = (int)($_POST['booking_id'] ?? 0);
    $did   = (int)($_POST['driver_id'] ?? 0);
    $amt   = (float)($_POST['amount'] ?? 0);
    $when  = (string)($_POST['paid_at'] ?? '');
    $meth  = trim((string)($_POST['method'] ?? 'Bank Transfer'));
    $ref   = trim((string)($_POST['reference'] ?? ''));
    $notes = trim((string)($_POST['notes'] ?? ''));

    if ($bid <= 0 || $did <= 0) throw new RuntimeException('Invalid booking/driver.');
    if ($amt <= 0)              throw new RuntimeException('Amount must be positive.');
    if ($when === '')           $when = date('Y-m-d H:i:s');

    $bk         = booking_driver_due($cid, $bid);
    $bookingRef = $bk['booking_ref'];
    $driverDue  = (float)$bk['driver_due'];
    if ($driverDue <= 0) throw new RuntimeException('No driver amount due for this booking.');

    $paidSoFar   = driver_paid_sum($cid, $bid);
    $outstanding = max(0.0, round($driverDue - $paidSoFar, 2));
    if ($outstanding <= 0) throw new RuntimeException('Nothing outstanding for this booking.');
    if ($amt > $outstanding) throw new RuntimeException('Amount exceeds outstanding (£'.number_format($outstanding,2).').');

    $pdo = db();
    $pdo->beginTransaction();

    $ins = $pdo->prepare("
      INSERT INTO driver_payouts
        (company_id, booking_id, driver_id, amount, paid_at, method, reference, notes, created_at)
      VALUES
        (:cid, :bid, :did, :amt, :paid_at, :meth, :ref, :notes, NOW())
    ");
    $ins->execute([
      ':cid'=>$cid, ':bid'=>$bid, ':did'=>$did, ':amt'=>$amt, ':paid_at'=>$when,
      ':meth'=>$meth !== '' ? $meth : null,
      ':ref' =>$ref  !== '' ? $ref  : null,
      ':notes'=>$notes !== '' ? $notes : null,
    ]);
    $payoutId = (int)$pdo->lastInsertId();

    if (!tbl_exists('transactions')) throw new RuntimeException('transactions table missing.');
    $tx = $pdo->prepare("
      INSERT INTO transactions
        (company_id, date, account_code, segment, type, amount, booking_id, reference, notes, created_at)
      VALUES
        (:cid, DATE(:paid_at), 'DRIVER_PAYOUT', 'Chauffeur', 'expense',
         :amt, :bid, :ref, :notes, NOW())
    ");
    $tx->execute([
      ':cid'=>$cid,
      ':paid_at'=>$when,
      ':amt'=>$amt,
      ':bid'=>$bid,
      ':ref'=> $ref !== '' ? $ref : ('Driver payout for '.$bookingRef),
      ':notes'=>$notes !== '' ? $notes : null,
    ]);
    if ($tx->rowCount() !== 1) throw new RuntimeException('Accounting entry not recorded.');

    $newPaid  = round($paidSoFar + $amt, 2);
    $balance  = max(0.0, round($driverDue - $newPaid, 2));

    if (tbl_exists('pending_payments')) {
      $del = $pdo->prepare("DELETE FROM pending_payments WHERE company_id=:cid AND booking_id=:bid AND type='driver'");
      $del->execute([':cid'=>$cid, ':bid'=>$bid]);

      if ($balance > 0) {
        $pend = $pdo->prepare("
          INSERT INTO pending_payments
            (company_id, booking_id, booking_ref, `type`, total_due, amount_paid, balance_due, status, created_at)
          VALUES
            (:cid,:bid,:ref,'driver',:total,:paid,:bal,'pending',NOW())
        ");
        $pend->execute([
          ':cid'=>$cid,
          ':bid'=>$bid,
          ':ref'=>$bookingRef,
          ':total'=>round($driverDue,2),
          ':paid' =>$newPaid,
          ':bal'  =>$balance,
        ]);
      }
    }

    audit_log('payout.driver.create','booking',$bid,[
      'driver_id'=>$did,'amount'=>$amt,'paid_at'=>$when,'method'=>$meth,'reference'=>$ref,
      'total_due'=>round($driverDue,2),'paid_after'=>$newPaid,'balance_remaining'=>$balance
    ]);

    $pdo->commit();

    $qsBack = http_build_query([
      'from'=>$from,'to'=>$to,'status'=>$status,'q'=>$q,
      'driver_name'=>$driverName,'job6'=>$job6,'out_min'=>$outMinStr,'out_max'=>$outMaxStr
    ]);
    $returnLink = 'drivers.php?'.$qsBack;
    $receiptUrl = url_modules('payments/driver_receipt.php')
                . '?payout_id='.(int)$payoutId
                . '&return='.urlencode($returnLink);
    redirect($receiptUrl);

  } catch (Throwable $e) {
    if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
    $errors[] = (defined('APP_ENV') && APP_ENV === 'dev') ? $e->getMessage() : 'Unable to record payout.';
  }
}

if (isset($_GET['ok'])) $notice = 'Payout recorded.';

/* ---------------- Render page ---------------- */
include dirname(__DIR__, 2) . '/includes/header.php';
?>
<style>
  /* Layout fixes for consistent alignment across rows */
  .pay-table { table-layout: fixed; }
  .pay-table th, .pay-table td { vertical-align: middle; }

  /* Column widths (tweak as needed) */
  .pay-col-ref     { width: 18%; }
  .pay-col-client  { width: 18%; }
  .pay-col-pickup  { width: 16%; }
  .pay-col-driver  { width: 24%; }
  .pay-col-money   { width: 10%; }
  .pay-col-actions { width: 14%; min-width: 200px; }

  /* Keep controls on one line */
  .btn-group-nw { display: inline-flex; gap: .25rem; flex-wrap: nowrap; }
  .nowrap { white-space: nowrap; }

  /* Driver cell formatting */
  .driver-line { display: inline-flex; align-items: center; gap: .5rem; flex-wrap: nowrap; width: 100%; }
  .driver-name { display: inline-block; max-width: 140px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; }
</style>

<div class="d-flex justify-content-between align-items-center mb-3">
  <div>
    <h1 class="h4 mb-0">Driver Payments</h1>
    <div class="text-muted">Cleared bookings only (Completed). Filter by date, driver, job #, outstanding or search.</div>
  </div>
  <div class="d-flex flex-wrap gap-2">
    <?php $exportQs = http_build_query([
      'from'=>$from,'to'=>$to,'status'=>$status,'q'=>$q,
      'driver_name'=>$driverName,'job6'=>$job6,'out_min'=>$outMinStr,'out_max'=>$outMaxStr
    ]); ?>
    <a class="btn btn-outline-secondary" href="<?= e(url_modules('dashboard/md.php')) ?>">← MD Dashboard</a>
    <a class="btn btn-outline-primary" href="<?= e(url_modules('payments/export_csv.php').'?'.$exportQs) ?>">⬇️ Export CSV</a>
  </div>
</div>

<?php if ($notice): ?><div class="alert alert-success"><?= e($notice) ?></div><?php endif; ?>
<?php if ($errors): ?>
  <div class="alert alert-danger"><ul class="mb-0"><?php foreach ($errors as $er) echo '<li>'.e($er).'</li>'; ?></ul></div>
<?php endif; ?>

<div class="card shadow-sm mb-3">
  <div class="card-body">
    <!-- Compact two-row filter UI -->
    <form class="row row-cols-1 row-cols-md-6 g-2 align-items-end" method="get">
      <div class="col">
        <label class="form-label small">From</label>
        <input type="date" class="form-control form-control-sm" name="from" value="<?= e($from) ?>">
      </div>
      <div class="col">
        <label class="form-label small">To</label>
        <input type="date" class="form-control form-control-sm" name="to" value="<?= e($to) ?>">
      </div>
      <div class="col">
        <label class="form-label small">Status</label>
        <select class="form-select form-select-sm" name="status">
          <option value="unpaid" <?= $status==='unpaid'?'selected':'' ?>>Unpaid</option>
          <option value="paid"   <?= $status==='paid'  ?'selected':'' ?>>Paid</option>
          <option value="all"    <?= $status==='all'   ?'selected':'' ?>>All</option>
        </select>
      </div>
      <div class="col">
        <label class="form-label small">Driver (partial)</label>
        <input class="form-control form-control-sm" name="driver_name" placeholder="e.g. Ali / Khan" value="<?= e($driverName) ?>">
      </div>
      <div class="col">
        <label class="form-label small">Job # (last 6)</label>
        <input class="form-control form-control-sm" name="job6" maxlength="6" pattern="\d{0,6}" value="<?= e($job6) ?>">
      </div>

      <div class="col">
        <label class="form-label small">Outstanding £ Min</label>
        <input type="number" step="0.01" min="0" class="form-control form-control-sm" name="out_min" value="<?= e($outMinStr) ?>">
      </div>
      <div class="col">
        <label class="form-label small">Outstanding £ Max</label>
        <input type="number" step="0.01" min="0" class="form-control form-control-sm" name="out_max" value="<?= e($outMaxStr) ?>">
      </div>

      <div class="col d-grid">
        <button class="btn btn-sm btn-primary">Filter</button>
      </div>
      <div class="col d-grid">
        <a class="btn btn-sm btn-outline-secondary" href="<?= e(url_modules('payments/drivers.php')) ?>">Clear</a>
      </div>
    </form>
  </div>
</div>

<div class="card shadow-sm">
  <div class="table-responsive">
    <table class="table align-middle mb-0 pay-table">
      <thead class="table-light">
        <tr>
          <th class="pay-col-ref">Ref</th>
          <th class="pay-col-client">Client</th>
          <th class="pay-col-pickup">Pickup</th>
          <th class="pay-col-driver">Driver</th>
          <th class="text-end pay-col-money">£ Driver Due</th>
          <th class="text-end pay-col-money">£ Paid</th>
          <th class="text-end pay-col-money">£ Outstanding</th>
          <th class="text-end pay-col-actions">Actions</th>
        </tr>
      </thead>
      <tbody>
        <?php if ($rows): foreach ($rows as $r):
          $due    = (float)$r['driver_due'];
          $paid   = (float)$r['paid_sum'];
          $out    = max(0, $due - $paid);
          $badge  = $out > 0 ? 'danger' : 'success';
          $pdt    = trim(($r['pickup_date'] ?? '') . ' ' . ($r['pickup_time'] ?? ''));
          $latestPayoutId = (int)($r['latest_payout_id'] ?? 0);
          $returnQs = http_build_query([
            'from'=>$from,'to'=>$to,'status'=>$status,'q'=>$q,
            'driver_name'=>$driverName,'job6'=>$job6,'out_min'=>$outMinStr,'out_max'=>$outMaxStr
          ]);
          $receiptLink = $latestPayoutId > 0
            ? url_modules('payments/driver_receipt.php') . '?payout_id=' . $latestPayoutId . '&return=' . urlencode('drivers.php?'.$returnQs)
            : '';
        ?>
        <tr class="<?= $out > 0 ? '' : 'table-success' ?>">
          <td class="pay-col-ref"><strong><?= e($r['booking_ref'] ?? ('#'.$r['id'])) ?></strong></td>
          <td class="pay-col-client"><?= e($r['client_name'] ?? '—') ?></td>
          <td class="pay-col-pickup"><?= e($pdt) ?></td>
          <td class="pay-col-driver">
            <?php if (!empty($r['driver_id'])): ?>
              <div class="driver-line">
                <span class="driver-name"><?= e($r['driver_name'] ?? '—') ?></span>
                <div class="btn-group-nw">
                  <a class="btn btn-sm btn-outline-secondary"
                     href="<?= e(url_modules('drivers/view.php')) . '?id=' . (int)$r['driver_id'] ?>"
                     target="_blank" rel="noopener">Driver Profile</a>
                  <?php if (!empty($r['driver_phone'])): ?>
                    <a class="btn btn-sm btn-outline-secondary" href="tel:<?= e($r['driver_phone']) ?>">Call</a>
                  <?php endif; ?>
                </div>
              </div>
            <?php else: ?>
              <span class="text-muted">No driver</span>
            <?php endif; ?>
          </td>
          <td class="text-end pay-col-money">£<?= number_format($due, 2) ?></td>
          <td class="text-end pay-col-money">£<?= number_format($paid, 2) ?></td>
          <td class="text-end pay-col-money"><span class="badge text-bg-<?= e($badge) ?>">£<?= number_format($out, 2) ?></span></td>
          <td class="text-end pay-col-actions">
            <div class="btn-group-nw nowrap justify-content-end">
              <!-- View Job -->
              <button type="button"
                      class="btn btn-sm btn-outline-primary js-job"
                      data-booking-id="<?= (int)$r['id'] ?>">
                View Job
              </button>

              <?php if (!empty($r['driver_id']) && $out > 0): ?>
                <!-- Mark Paid -->
                <button type="button"
                        class="btn btn-sm btn-dark js-pay"
                        data-booking-id="<?= (int)$r['id'] ?>"
                        data-driver-id="<?= (int)$r['driver_id'] ?>"
                        data-outstanding="<?= number_format($out, 2, '.', '') ?>"
                        data-ref="<?= e($r['booking_ref'] ?? ('#'.$r['id'])) ?>"
                        data-price="<?= number_format((float)$r['driver_price'], 2, '.', '') ?>"
                        data-park="<?= number_format((float)$r['driver_parking'], 2, '.', '') ?>"
                        data-wait="<?= number_format((float)$r['driver_waiting'], 2, '.', '') ?>">
                  Mark Paid
                </button>
              <?php endif; ?>

              <?php if ($paid > 0 && $latestPayoutId > 0): ?>
                <!-- Print opens the latest driver receipt -->
                <a class="btn btn-sm btn-outline-success"
                   href="<?= e($receiptLink) ?>"
                   target="_blank" rel="noopener">
                  Print
                </a>
              <?php endif; ?>
            </div>
          </td>
        </tr>
        <?php endforeach; else: ?>
          <tr><td colspan="8" class="text-center text-muted py-4">No results.</td></tr>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>

<!-- Mark as Paid Modal -->
<div class="modal fade" id="payModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <form class="modal-content" method="post">
      <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
      <input type="hidden" name="action" value="mark_paid">
      <input type="hidden" name="booking_id" id="pay_booking_id">
      <input type="hidden" name="driver_id"  id="pay_driver_id">
      <div class="modal-header">
        <h5 class="modal-title">Mark Driver Payout</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div class="mb-2 small text-muted" id="pay_ref"></div>

        <div class="border rounded p-2 mb-3 bg-light">
          <div class="row g-2">
            <div class="col-4"><div class="small text-muted">Driver Price</div><div id="bd_price" class="fw-semibold">£0.00</div></div>
            <div class="col-4"><div class="small text-muted">Parking</div><div id="bd_park" class="fw-semibold">£0.00</div></div>
            <div class="col-4"><div class="small text-muted">Waiting</div><div id="bd_wait" class="fw-semibold">£0.00</div></div>
          </div>
          <div class="mt-1 small text-muted">Default Amount = Outstanding</div>
        </div>

        <div class="row g-3">
          <div class="col-6">
            <label class="form-label">Amount (£)</label>
            <input type="number" step="0.01" min="0.01" class="form-control" name="amount" id="pay_amount" required>
          </div>
          <div class="col-6">
            <label class="form-label">Paid At</label>
            <input type="datetime-local" class="form-control" name="paid_at" value="<?= e(date('Y-m-d\TH:i')) ?>">
          </div>
          <div class="col-6">
            <label class="form-label">Method</label>
            <input class="form-control" name="method" placeholder="Bank Transfer / Cash / Card" value="Bank Transfer">
          </div>
          <div class="col-6">
            <label class="form-label">Reference</label>
            <input class="form-control" name="reference" placeholder="Txn ID / Note">
          </div>
          <div class="col-12">
            <label class="form-label">Notes (optional)</label>
            <textarea class="form-control" name="notes" rows="2"></textarea>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-dark">Save Payout</button>
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
      </div>
    </form>
  </div>
</div>

<!-- Job Details Modal -->
<div class="modal fade" id="jobModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Job Details</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div id="job_content" class="small">Loading…</div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<?php include dirname(__DIR__, 2) . '/includes/footer.php'; ?>

<script>
(function(){
  const payModal  = new bootstrap.Modal(document.getElementById('payModal'));
  const jobModal  = new bootstrap.Modal(document.getElementById('jobModal'));

  // Open payout modal with breakdown
  document.addEventListener('click', function(e){
    const btn = e.target.closest('.js-pay');
    if (!btn) return;
    const bid  = btn.getAttribute('data-booking-id');
    const did  = btn.getAttribute('data-driver-id');
    const out  = btn.getAttribute('data-outstanding') || '0.00';
    const ref  = btn.getAttribute('data-ref') || '';
    const price = btn.getAttribute('data-price') || '0.00';
    const park  = btn.getAttribute('data-park')  || '0.00';
    const wait  = btn.getAttribute('data-wait')  || '0.00';

    document.getElementById('pay_booking_id').value = bid;
    document.getElementById('pay_driver_id').value  = did;
    document.getElementById('pay_amount').value     = out;
    document.getElementById('pay_ref').textContent  = 'Booking: ' + ref;

    document.getElementById('bd_price').textContent = '£' + Number(price).toFixed(2);
    document.getElementById('bd_park').textContent  = '£' + Number(park).toFixed(2);
    document.getElementById('bd_wait').textContent  = '£' + Number(wait).toFixed(2);

    payModal.show();
  });

  // Open Job modal and fetch details
  document.addEventListener('click', function(e){
    const btn = e.target.closest('.js-job');
    if (!btn) return;
    const bid = btn.getAttribute('data-booking-id');
    const box = document.getElementById('job_content');
    box.innerHTML = 'Loading…';

    fetch('drivers.php?fetch_job=1&booking_id=' + encodeURIComponent(bid), {
      headers: {'X-Requested-With':'XMLHttpRequest'}
    })
    .then(r => r.text())
    .then(html => { box.innerHTML = html; })
    .catch(() => { box.innerHTML = '<div class="text-danger">Failed to load.</div>'; });

    jobModal.show();
  });
})();
</script>
